/*
  NewsMaster / PrintMaster shapes library tool
  (c) SysTools 2019
  http://systools.losthost.org/

  Both shapes format supported: original (extract, append) and
  from CAPTURE.EXE tool (extract only).

  All [u]int<8/16/32>_t types declaration taken out from
  <stdint.h> header file which not existed yet back in 1988
  and also BITMAPFILEHEADER and BITMAPINFOHEADER declarations
  which was taken out from the <windows.h> header file.

  Note that 32bit types may need tweaking for 64bit compilers
  since "long" are the only 32bit type on 16bit compiler.

  This tool utilizes only static buffers for all the work -
  no memory allocated or freed in the code below.

  Borland Turbo C Compiler 2.01 (DOS16)
  TCC -w -g1 -O -Z shapelib.c

  GNU C Compiler 3.2 (Win32)
  GCC -Wall -pedantic -Werror -Os -s -o shapelib shapelib.c
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <io.h>

/* required types, DOS16 / Win32 compatible */
typedef   signed char   int8_t;
typedef unsigned char  uint8_t;
typedef   signed short  int16_t;
typedef unsigned short uint16_t;
/* note: x32 bit types may need tweaking for x64 compilation */
typedef   signed long   int32_t;
typedef unsigned long  uint32_t;

#define BMP_TYPE_BM 0x4D42
#define SDR_NAME_SZ 16

#pragma pack(push, 1)
typedef struct {
  uint16_t bfType;
  uint32_t bfSize;
  uint16_t bfReserved1;
  uint16_t bfReserved2;
  uint32_t bfOffBits;
} BITMAPFILEHEADER;

typedef struct {
  uint32_t biSize;
  int32_t  biWidth;
  int32_t  biHeight;
  uint16_t biPlanes;
  uint16_t biBitCount;
  uint32_t biCompression;
  uint32_t biSizeImage;
  int32_t  biXPelsPerMeter;
  int32_t  biYPelsPerMeter;
  uint32_t biClrUsed;
  uint32_t biClrImportant;
} BITMAPINFOHEADER;

typedef struct {
  BITMAPFILEHEADER hdr_file;
  BITMAPINFOHEADER hdr_info;
  uint32_t         hdr_cpal[256];
} bmp_head;

typedef struct {
  uint32_t zero;
  int16_t  coffx;
  int16_t  coffy;
  int16_t  left;
  int16_t  top;
  int16_t  right;
  int16_t  bottom;
  int16_t  unused;
  uint32_t size;
} cpt_head;

typedef struct {
  uint8_t  rowsize;
  uint8_t  height;
  uint16_t width;
} shp_head;

typedef struct {
  char name[SDR_NAME_SZ];
} sdr_item;

typedef struct {
  uint32_t offs;
} sdx_item;
#pragma pack(pop)

/* set fixed limit for max images per one library */
#define MAX_ITEM 200
/* set shp max supported width and height */
#define SHP_MAX_WIDTH  960U
#define SHP_MAX_HEIGHT 255U
/* set cpt max supported width and height */
#define CPT_MAX_WIDTH  640U
#define CPT_MAX_HEIGHT 480U
/* max image data size in bytes 960x255 (30600 bytes) */
#define MAX_DATA_SHP (((SHP_MAX_WIDTH + 7) / 8) * SHP_MAX_HEIGHT)
/* cpt format can hold larger images and max resolution where
   CAPTURE.EXE still can be used are 640x480 (38400 bytes) */
#define MAX_DATA_CPT (((CPT_MAX_WIDTH + 7) / 8) * CPT_MAX_HEIGHT)
/* max of two buffers */
#define MAX_DATA MAX_DATA_CPT
/* file types index in 4-char string with extensions */
#define FILE_TYPE_SHP 0
#define FILE_TYPE_SDR 1
#define FILE_TYPE_SDX 2
/* library file extensions */
const static char ext_list[] = ".SHP\0.SDR\0.SDX";
/* buffer for max offs data */
static sdx_item sdx_list[MAX_ITEM];
/* buffer for max names data */
static sdr_item sdr_list[MAX_ITEM];
/* buffer for max image data */
static uint8_t data[MAX_DATA];
/* bitmap file headers */
static bmp_head bmp_data;
/* global library filename for open (DOS 8.3 ASCIIZ) */
static char lib_name[13];
/* max items in shape library */
static int count;

/* calculate BMP image size */
static int32_t CalcBMPSize(int32_t w, int32_t h, uint16_t b) {
  w = ((((w * b) + 31) & ~31) >> 3);
  w *= (h < 0) ? (-h) : h;
  return(w);
}

/* universal bitmap header initialization routine */
static uint32_t InitBMPHeaders(bmp_head *bh, int32_t w, int32_t h, uint16_t b) {
uint32_t l;
  /* header size */
  l =
    /* file hader */
    sizeof(bh->hdr_file) +
    /* bitmap header */
    sizeof(bh->hdr_info) +
    /* palette size for index color images */
    ((b <= 8) ? (sizeof(bh->hdr_cpal[0]) << b) : 0);
  if (bh) {
    memset(bh, 0, sizeof(bh[0]));
    bh->hdr_info.biSize = sizeof(bh->hdr_info);
    bh->hdr_info.biWidth = w;
    bh->hdr_info.biHeight = h;
    bh->hdr_info.biPlanes = 1;
    bh->hdr_info.biBitCount = b;
    bh->hdr_info.biSizeImage = CalcBMPSize(w, h, b);
    bh->hdr_file.bfType = BMP_TYPE_BM;
    bh->hdr_file.bfOffBits = l;
    bh->hdr_file.bfSize = l + bh->hdr_info.biSizeImage;
  }
  return(l);
}

/* merge strings list to the single string with the spaces between */
static void StrListMerge(char *buff, int bmax, char *list[], int lmax) {
int i, n;
char *s;
  memset(buff, 0, bmax * sizeof(buff[0]));
  n = 0;
  for (i = 0; i < lmax; i++) {
    for (s = list[i]; *s; s++) {
      if (n == bmax) { break; }
      buff[n] = *s;
      n++;
    }
    if (n == bmax) { break; }
    buff[n] = ' ';
    n++;
  }
  /* add null at string end */
  if (n) {
    n--;
    buff[n] = 0;
  }
  /* trim space at the end if there is not enough memory for next word */
  if (n && (buff[n - 1] == ' ')) {
    n--;
    buff[n] = 0;
  }
}

/* prepare all global static buffers before using */
static void InitData(void) {
  count = 0;
  memset(data, 0, MAX_DATA);
  memset(&bmp_data, 0, sizeof(bmp_data));
  memset(sdx_list, 0, MAX_ITEM * sizeof(sdx_list[0]));
  memset(sdr_list, 0, MAX_ITEM * sizeof(sdr_list[0]));
}

/* add extension to the base name of the file */
static char *BuildName(char *library, int type) {
int i;
  /* in case of extension - cut it (DOS 8.3 filename) */
  for (i = 0; i < 8; i++) {
    /* end of the string or dot from extension */
    if ((!library[i]) || (library[i] == '.')) { break; }
  }
  /* copy name */
  memcpy(lib_name, library, i * sizeof(library[0]));
  /* add required extension */
  memcpy(&lib_name[i], &ext_list[type * 5], 5 * sizeof(library[0]));
  /* return pointer for easy use in open() since it's global anyway */
  return(lib_name);
}

/* file size helper routine */
static uint32_t FileSize(int fl) {
uint32_t p, l;
  p = tell(fl);
  lseek(fl, 0, SEEK_END);
  l = tell(fl);
  lseek(fl, p, SEEK_SET);
  return(l);
}

/* checks if current position in file points to the start of the shp item */
static int IsItemSHP(int fl, uint32_t sz, shp_head *shp) {
uint32_t l;
  /* save current position if invalid format */
  l = tell(fl);
  /* not enough space in file for header */
  if (read(fl, shp, sizeof(shp[0])) != sizeof(shp[0])) { shp = NULL; }
  /* width or height are zero or width more than max or invalid rowsize */
  if (shp && ((!shp->width) || (!shp->height) ||
    (shp->width > SHP_MAX_WIDTH) ||
    (((shp->width + 7) / 8) != shp->rowsize)
  )) { shp = NULL; }
  /* check that file has enough space to hold the whole image */
  if (shp && sz && ((l + sizeof(shp[0]) + (shp->rowsize * shp->height) + 1) > sz)) {
    shp = NULL;
  }
  /* invalid format - rewind back file postion */
  if (!shp) { lseek(fl, l, SEEK_SET); }
  return(shp ? 1 : 0);
}

/* checks if current position in file points to the start of the cpt item */
static int IsItemCPT(int fl, uint32_t sz, cpt_head *cpt) {
uint32_t l;
  /* save current position if invalid format */
  l = tell(fl);
  /* not enough space in file for header */
  if (read(fl, cpt, sizeof(cpt[0])) != sizeof(cpt[0])) { cpt = NULL; }
  /* must starts with 4 zero bytes to tell apart from the shp header */
  if (cpt && cpt->zero) { cpt = NULL; }
  /* calc width and height, note that there is no +1 here
     like in some other formats (i.e. width = x2 - x1 + 1) */
  if (cpt) {
    cpt->right -= cpt->left; /* width */
    cpt->bottom -= cpt->top; /* height */
    /* just in case */
    cpt->left = 0;
    cpt->top = 0;
  }
  if (cpt && (
    /* width and height can't be negative */
    (cpt->right <= 0) || (cpt->bottom <= 0) ||
    /* or bigger than max */
    (cpt->right > CPT_MAX_WIDTH) || (cpt->bottom > CPT_MAX_HEIGHT) ||
    /* coeffs can't be zero or less */
    (cpt->coffx <= 0) || (cpt->coffy <= 0) ||
    /*
      explanation for this weird check:
      NewsMaster II recalculate image screen dimensions by the next formula:
        width  = ((right - left) * 360) / coffx
        height = ((bottom -  up) * 360) / coffy
      if after division result will be too big to fit in word with sign (07FFFh)
      the program will crash with the fatal error "divide by zero"
      07FFFh = 32767, so let's see what's need to be checked:
        32767 >= (dimension * 360) / coff
        coff * 32767 >= dimension * 360
        (coff * 32767) / 360 >= dimension
        coff * 91 >= dimension
        coff >= dimension / 91
    */
    (cpt->coffx < (cpt->right / 91)) || (cpt->coffy < (cpt->bottom / 91)) ||
    /* check the correct whole image size */
    ((((cpt->right + 7) / 8) * cpt->bottom) != cpt->size)
  )) { cpt = NULL; }
  /* check that file has enough space to hold the whole image */
  if (cpt && sz && ((l + sizeof(cpt[0]) + cpt->size) > sz)) { cpt = NULL; }
  /* invalid format - rewind back file postion */
  if (!cpt) { lseek(fl, l, SEEK_SET); }
  return(cpt ? 1 : 0);
}

static int IsFileBMP(int fl, uint32_t sz, bmp_head *bmp) {
int32_t h;
uint32_t l;
  /* bitmap header size */
  l = InitBMPHeaders(NULL, 0, 0, 1);
  /* not enough space in file for header */
  if (read(fl, bmp, (int) l) != l) { bmp = NULL; }
  /* negative height */
  h = bmp->hdr_info.biHeight;
  h = (h < 0) ? (-h) : h;
  /* check BMP format */
  if (bmp && (
    /* check signature */
    (bmp->hdr_file.bfType != BMP_TYPE_BM) ||
    /* disk file can't be less than size in header (but fine if bigger) */
    (bmp->hdr_file.bfSize > sz) ||
    /* check v3 header */
    (bmp->hdr_info.biSize != sizeof(bmp->hdr_info)) ||
    /* check width and height */
    (bmp->hdr_info.biWidth <= 0) || (!h) ||
    /* and not exceed max allowed image resolution in shp format */
    (bmp->hdr_info.biWidth > SHP_MAX_WIDTH) || (h > SHP_MAX_HEIGHT) ||
    /* planes must be 1 */
    (bmp->hdr_info.biPlanes != 1) ||
    /* BPP must be 1 */
    (bmp->hdr_info.biBitCount != 1) ||
    /* compression unsupported */
    (bmp->hdr_info.biCompression)
  )) { bmp = NULL; }
  /* now the tricky part: since biSizeImage can be 0
     for non-compressed images - recalc it */
  if (bmp) {
    bmp->hdr_info.biSizeImage = CalcBMPSize(
      bmp->hdr_info.biWidth,
      bmp->hdr_info.biHeight,
      bmp->hdr_info.biPlanes
    );
  }
  /* last check for correct offset to the image */
  if (bmp && (
    /* in theory image can start inside header but it's weird */
    (bmp->hdr_file.bfOffBits < l) ||
    /* there should be enough space at image offset to hold image data */
    ((bmp->hdr_file.bfOffBits + bmp->hdr_info.biSizeImage) > sz)
  )) { bmp = NULL; }
  /* no need to rewind file ponter on error since testing standalone BMP file */
  return(bmp ? 1 : 0);
}

/* load data from library file */
static void LoadData(char *library) {
shp_head shp;
cpt_head cpt;
uint32_t sz;
int fl, i, j;
char c;
  /* open shapes library */
  fl = open(BuildName(library, FILE_TYPE_SHP), O_RDONLY | O_BINARY);
  if (fl != -1) {
    /* get file size */
    sz = FileSize(fl);
    while ((tell(fl) < sz) && (count < MAX_ITEM)) {
      sdx_list[count].offs = tell(fl);
      /* read as SHP format first (smaller header) */
      if (IsItemSHP(fl, sz, &shp)) {
        /* skip image data */
        lseek(fl, tell(fl) + (shp.rowsize * shp.height) + 1, SEEK_SET);
      } else {
        /* read as CPT format next (bigger header) */
        if (IsItemCPT(fl, sz, &cpt)) {
          /* skip image data */
          lseek(fl, tell(fl) + cpt.size, SEEK_SET);
        } else {
          /* something wrong - break */
          break;
        }
      }
      count++;
    }
    /* for append */
    if (count < MAX_ITEM) {
      sdx_list[count].offs = tell(fl);
    }
    close(fl);
  }
  /* at least one image exists */
  if (count) {
    /* open descriptions file */
    fl = open(BuildName(library, FILE_TYPE_SDR), O_RDONLY | O_BINARY);
    if (fl != -1) {
      /* calc approximate amount items in file */
      sz = FileSize(fl) / sizeof(sdr_list[0]);
      /* can't be more than total items */
      sz = (sz > count) ? count : sz;
      /* read items descriptions */
      read(fl, sdr_list, ((int) sz) * sizeof(sdr_list[0]));
      close(fl);
    }
    /* fix item names */
    c = 0;
    for (i = 0; i < count; i++) {
      /* end reached if 0x00 or 0x1A as in NewsMaster loader code */
      if ((sdr_list[i].name[0] == 0x00) || (sdr_list[i].name[0] == 0x1A)) { c = 1; }
      /* clear text items from junk bytes after zero tail char */
      if (!c) {
        c = 1;
        for (j = 0; j < SDR_NAME_SZ; j++) {
          sdr_list[i].name[j] *= c;
          c &= sdr_list[i].name[j] ? 1 : 0;
        }
        c = 0;
      } else {
        /* clear tail junk chars if any here from the sdr file */
        memset(sdr_list[i].name, 0, SDR_NAME_SZ * sizeof(sdr_list[0].name[0]));
        /* fill with same default names as in NewsMaster: #001, #002, ... */
        sprintf(sdr_list[i].name, "#%03d", i + 1);
      }
    }
  }
}

/* output item names with thier order number inside library */
static void ListData(void) {
char s[SDR_NAME_SZ + 1];
int i;
  s[SDR_NAME_SZ] = 0;
  for (i = 0; i < count; i++) {
    memcpy(s, sdr_list[i].name, SDR_NAME_SZ * sizeof(s[0]));
    printf("%03u-%s\n", i + 1, s);
  }
}

/* extract shape by number */
static void ExtractShape(char *library, int n) {
char name[SDR_NAME_SZ + 1];
shp_head shp;
cpt_head cpt;
int fl, i;
  /* open shapes library */
  fl = open(BuildName(library, FILE_TYPE_SHP), O_RDONLY | O_BINARY);
  if (fl != -1) {
    lseek(fl, sdx_list[n - 1].offs, SEEK_SET);
    /* detect image format */
    if (!IsItemCPT(fl, 0, &cpt)) {
      if (IsItemSHP(fl, 0, &shp)) {
        /* unification for later usage */
        memset(&cpt, 0, sizeof(cpt));
        cpt.right = shp.width;
        cpt.bottom = shp.height;
        cpt.size = shp.rowsize * shp.height;
      } else {
        /* error */
        cpt.size = 0;
      }
    }
    if (cpt.size <= MAX_DATA) {
      /* read data */
      read(fl, data, (int) cpt.size);
    } else {
      /* error */
      cpt.size = 0;
    }
    close(fl);
    /* no error */
    if (cpt.size) {
      /* output shape name */
      memcpy(name, sdr_list[n - 1].name, SDR_NAME_SZ * sizeof(name[0]));
      name[SDR_NAME_SZ] = 0;
      printf("%03u|%s -> ", n, name);
      /* generate output file name */
      sprintf(name, "SHAPE%03u.BMP", n);
      printf("%s\n", name);
      /* create output file */
      fl = open(name, O_RDWR | O_BINARY | O_CREAT | O_TRUNC, S_IREAD | S_IWRITE);
      if (fl != -1) {
        /* invert image colors for BMP format */
        for (i = 0; i < cpt.size; i++) {
          data[i] = ~data[i];
        }
        /* init bitmap headers */
        InitBMPHeaders(&bmp_data, cpt.right, cpt.bottom, 1);
        /* add white palette color */
        bmp_data.hdr_cpal[1] = 0xFFFFFFUL;
        /* actual headers size (include palette) will be in bfOffBits */
        write(fl, &bmp_data, (int) bmp_data.hdr_file.bfOffBits);
        /* row size */
        cpt.unused = cpt.size / cpt.bottom;
        /* for padding */
        cpt.zero = 0;
        /* padding size */
        n = (4 - (cpt.unused % 4)) % 4;
        /* write image data by row (bottom-top image) */
        while (cpt.bottom--) {
          /* write image row */
          write(fl, &data[cpt.bottom * cpt.unused], cpt.unused);
          /* write bmp padding bytes */
          write(fl, &cpt.zero, n);
        }
        close(fl);
        printf("\ndone\n\n");
      } else {
        printf("Error: can't create output file.\n\n");
      }
    } else {
      printf("Error: invalid shape or too big to fit in internal buffer.\n\n");
    }
  }
}

/* append new shape to the end of the library */
static void AppendShape(char *library, char *image, char *name) {
uint16_t lb;
shp_head shp;
int fl, i, h;
  /* copy item name (note that offset in sdx_list[] was init above) */
  memcpy(sdr_list[count].name, name, SDR_NAME_SZ * sizeof(name[0]));
  printf("%03u|%s <- %s\n", count + 1, name, image);
  fl = open(image, O_RDONLY | O_BINARY);
  if (fl != -1) {
    /* check for correct BMP file */
    i = IsFileBMP(fl, FileSize(fl), &bmp_data);
    if (i) {
      /* row length (in bytes) for BMP */
      lb = CalcBMPSize(bmp_data.hdr_info.biWidth, 1, bmp_data.hdr_info.biBitCount);
      /* fill in shape header */
      shp.rowsize = (bmp_data.hdr_info.biWidth + 7) / 8;
      shp.width = bmp_data.hdr_info.biWidth;
      /* negative height */
      h = (int) bmp_data.hdr_info.biHeight;
      shp.height = (h < 0) ? (-h) : h;
      /* seek to the bitmap data start */
      lseek(fl, bmp_data.hdr_file.bfOffBits, SEEK_SET);
      /* read and convert BMP file to SHP */
      while (h) {
        /* instead of disk seeking for bottom-top images
           move pointer to correct address in memory */
        read(fl, &data[((h < 0) ? (shp.height + h) : (h - 1)) * shp.rowsize], shp.rowsize);
        /* skip BMP padding bytes */
        lseek(fl, lb - shp.rowsize, SEEK_CUR);
        /* negative (top-bottom) or positive (bottom-top) BMP file */
        h += (h < 0) ? 1 : (-1);
      }
      /* invert colors if needed */
      if (!bmp_data.hdr_cpal[0]) {
        for (i = 0; i < (shp.rowsize * shp.height); i++) {
          data[i] = ~data[i];
        }
      }
      /* no errors */
      i = 1;
    }
    /* now fl handle can be reused (only one opened file at a time) */
    close(fl);
    /* no errors? */
    if (i) {
      /* open file or create if new library */
      fl = count ?
           /* at least one image exists - open existing library */
           open(BuildName(library, FILE_TYPE_SHP), O_RDWR | O_BINARY) :
           /* no images in library - create new library file */
           open(BuildName(library, FILE_TYPE_SHP),
             O_RDWR | O_BINARY | O_CREAT | O_TRUNC, S_IREAD | S_IWRITE);
      if (fl != -1) {
        /* seek to the place after last image */
        lseek(fl, sdx_list[count].offs, SEEK_SET);
        /* write shape header */
        write(fl, &shp, sizeof(shp));
        /* write image data */
        write(fl, data, shp.rowsize * shp.height);
        /* write unknown byte */
        i = 0;
        write(fl, &i, 1);
        close(fl);
        /* always overwrite updated description and offset files */
        count++;
        /* description file */
        fl = open(BuildName(library, FILE_TYPE_SDR),
          O_RDWR | O_BINARY | O_CREAT | O_TRUNC, S_IREAD | S_IWRITE);
        if (fl) {
          write(fl, sdr_list, sizeof(sdr_list[0]) * count);
          close(fl);
        }
        /* offsets file */
        fl = open(BuildName(library, FILE_TYPE_SDX),
          O_RDWR | O_BINARY | O_CREAT | O_TRUNC, S_IREAD | S_IWRITE);
        if (fl) {
          write(fl, sdx_list, sizeof(sdx_list[0]) * count);
          close(fl);
        }
        printf("\ndone\n\n");
      } else {
        printf("Error: can't create or open for write shape library file.\n\n");
      }
    } else {
      printf(
        "Error: not a BMP file, invalid format or larger than %ux%u pixels.\n\n",
        SHP_MAX_WIDTH, SHP_MAX_HEIGHT
      );
    }
  } else {
    printf("Error: can't open input BMP file for read.\n\n");
  }
}

int main(int argc, char *argv[]) {
char s[SDR_NAME_SZ + 1];
int n;
  printf("NewsMaster / PrintMaster shapes library tool\n\n");
  if (argc < 2) {
    printf(
      "Usage: shapelib <library> [...]\n\n"
      "List images in library TEST.SHP (1 argument):\n"
      "  shapelib test\n\n"
      "Extract 4th image from library TEST.SHP (2 arguments):\n"
      "  shapelib test 4\n\n"
      "Append image to the library TEST.SHP (3 or more arguments):\n"
      "  shapelib test filename.bmp Item Description\n\n"
    );
    return(1);
  }
  /* init global structs */
  InitData();
  /* read library data */
  LoadData(argv[1]);
  /* decide what to do */
  switch (argc - 2) {
    case 0:
      if (count) {
        ListData();
      } else {
        printf("Error: can't open or invalid library file.\n\n");
      }
      break;
    case 1:
      if (count) {
        n = atoi(argv[2]);
        printf("Extract shape number \"%s\" (%d)...\n", argv[2], n);
        if ((n >= 1) && (n <= count)) {
          ExtractShape(argv[1], n);
        } else {
          printf("Error: invalid shape number, must be in range: 1 <= n <= %u.\n\n", count);
        }
      } else {
        printf("Error: can't open or invalid library file.\n\n");
      }
      break;
    /* >= 2 */
    default:
      if (count < MAX_ITEM) {
        /* merge item name from command line arguments */
        StrListMerge(s, SDR_NAME_SZ + 1, &argv[3], argc - 3);
        /* append image to the end of the current library */
        AppendShape(argv[1], argv[2], s);
      } else {
        printf("Error: too much shapes (>=%u) in this library, create a new one.\n\n", MAX_ITEM);
      }
      break;
  }
  return(0);
}
