; FASM source code
; http://flatassembler.net/
;
; prntodsk.asm
; (c) SysTools 2019
; http://systools.losthost.org/
;
; NewsMaster / PrintMaster virtual printer driver
; This printer driver tested and compatible with NewsMaster version 1.0, 1.5 and II.
; Set up any printer driver in NMCONFIG.EXE and replace NEWS.PRN file in the program
; directory with compiled binary file of this virtual printer driver.
; PRNTODSK.BIN => NEWS.PRN
;
; WARNING!
; Printer driver will be loaded via DOS Fn 04B03h (Load Overlay) to tiny static buffer.
; This buffer size only 1200 bytes long in NewsMaster 1.0.
; Because of that the driver must be as small as possible.
; For example:
; 1. Largest driver CGP220.PRN size is 1192 bytes and it's safe to use.
; 2. Driver in 1384 bytes long are already too big and will cause the program to crash.
; 3. Note that size of the driver IBMCMPCT.PRN is 1664 bytes (bigger than 1200) but it's
;    an .EXE file and amount of bytes it takes in the memory will be less than the file size.

; no "org 0100h" here - it's an overlay file
use16

; these jumps MUST be short jumps (2 bytes)
jmp @LnPitch8 ; 1
jmp @LnPitch6 ; 2
jmp @PrintRow ; 3
jmp @ResetAll ; 4
jmp @SkipPage ; 5
jmp @SkipLine ; 6

; call order for each page:
; [4] ResetAll
; [1] LnPitch8
; [3] PrintRow (for each row)
; [5] SkipPage

; -----------------------------------------------------------------------------

; and here long calls can be used

; stubs since this driver don't need it
@LnPitch8:
@LnPitch6:
@SkipLine:
  ; set zero flag as no error
  xor  ax, ax
  retf

@PrintRow:
  call @DumpFile
  retf

@ResetAll:
  call @OpenFile
  retf

@SkipPage:
  call @FreeFile
  retf

; -----------------------------------------------------------------------------

@OpenFile:
  ; init .BMP header
  xor  ax, ax
  ; file size = HDRLEN for now
  mov  [_bfSize_lo], HDRLEN
  mov  [_bfSize_hi], ax
  ; image height = 0
  mov  [_biHeight_lo], ax
  mov  [_biHeight_hi], ax
  ; image size = 0
  mov  [_biSizeImage_lo], ax
  mov  [_biSizeImage_hi], ax
  ; find first non-existent file
@OpenFile_find:
  ; next number
  inc  ax
  ; save it in the file handle variable
  mov  [_file], ax
  ; convert decimal number to text string
  mov  di, _name + 7
  mov  bx, 10d
  mov  cx, 4
@@:
  xor  dx, dx
  div  bx
  add  dl, '0'
  mov  [ds:di], dl
  dec  di
  loop @b
  ; try to get file attribute
  ; it's a check if file or a directory with that name already exists
  mov  dx, _name
  mov  al, 000h
  mov  ah, 043h
  int  021h
  ; error if carry flag is set (probably file not found)
  jc   @f
  xor  ax, ax
  ; make file handle invalid in case of overflow
  xchg ax, [_file]
  cmp  ax, 9999d
  ; if 9999 overflow - do not print anything
  jz   @OpenFile_ret
  ; or else - continue checking
  jmp  @OpenFile_find
@@:
  ; clear file handle for error - will be used as invalid handle
  ; since file handle 0 already preserved as standard input handle
  ; also cx - file attributes
  ; create new file
  xor  cx, cx
  mov  [_file], cx
  mov  dx, _name
  mov  ah, 03Ch
  int  021h
  jnc  @f
  ; zero flag will be non-empty for error
  mov  al, 001h
  jmp  @OpenFile_ret
@@:
  ; no error - save file handle for future use
  mov  [_file], ax
  ; write .BMP header
  mov  dx, _head
  mov  cx, HDRLEN
  mov  bx, ax
  mov  ah, 040h
  int  021h
  ; no error
  xor  ax, ax
@OpenFile_ret:
  or   ax, ax
  retn

; -----------------------------------------------------------------------------

@FreeFile:
  ; set as error
  mov  ax, 001h
  ; check for valid file handle
  cmp  [_file], 000h
  jz   @f
  ; seek to file start
  xor  cx, cx ; offset hi = 0
  xor  dx, dx ; offset lo = 0
  mov  bx, [_file] ; file handle
  mov  al, 000h ; seek from start
  mov  ah, 042h
  int  021h
  ; write updated .BMP header
  mov  dx, _head
  mov  cx, HDRLEN
  mov  bx, [_file]
  mov  ah, 040h
  int  021h
  ; close file
  mov  bx, [_file]
  mov  ah, 03Eh
  int  021h
  ; no error
  xor  ax, ax
  ; file handle not valid anymore
  mov  [_file], ax
@@:
  or   ax, ax
  retn

; -----------------------------------------------------------------------------

; input:
; es:si - image buffer address
; cx - size for image buffer in bytes
; al - non-zero if printing 0x0A at the end required after 0x0D
@DumpFile:
  ; set as error
  mov  ax, 001h
  ; check for valid file handle
  cmp  [_file], 000h
  jz   @DumpFile_ret
  ; revert image data
  ; note that each byte in B/W screen mode or a .BMP file represents horizonal line of 8 pixels
  ; but the data of a matrix printer is different: each byte represents a VERTICAL line of 8 dots
  ; because of that binary image data needs to be rearranged before it can be saved as bitmap
  mov  cl, 7
  @for_j: ; y = 0..7 also used as shift index
    ; save data pointer original value
    push si
    mov  di, _data
    mov  ch, 120d
    @for_i: ; x = 0..119
      xor  al, al
      mov  bl, 8
      @for_k: ; index in 8 byte block
        mov  ah, [es:si]
        ; since only cl register can be used as shift index value use it as for_j loop index too
        shr  ah, cl
        and  ah, 1
        shl  al, 1
        or   al, ah
        inc  si
        dec  bl
      jnz  @for_k
      ; note that in printer driver bit 1 means print (black ink) and 0 means skip (white paper)
      ; but in the .BMP file 0 are black and 1 are white, so bits in byte needs to be inverted
      not  al
      mov  [ds:di], al
      inc  di
      dec  ch
    jnz  @for_i
    ; save cx
    push cx
    mov  cx, 120d
    xor  ax, ax
    ; increment height
    ; do a decrement instead because this must be a negative value
    ; (see explanation below in .BMP header)
    sub  [_biHeight_lo], 001h
    sbb  [_biHeight_hi], ax
    ; update .BMP file headers
    add  [_bfSize_lo], cx
    adc  [_bfSize_hi], ax
    add  [_biSizeImage_lo], cx
    adc  [_biSizeImage_hi], ax
    ; write row of pixels to the file
    ; also cx - number of bytes to write
    mov  dx, _data
    mov  bx, [_file]
    mov  ah, 040h
    int  021h
    ; check for any key pressed
    call @CheckKey
    ; restore cx
    pop  cx
    ; rewind input data pointer to start
    pop  si
    ; test zero flag from CheckKey
    jnz  @DumpFile_ret
    ; next row
    dec  cl
  jns  @for_j
@DumpFile_ret:
  or  ax, ax
  retn

; -----------------------------------------------------------------------------

@CheckKey:
  ; check key in buffer
  mov  ax, 0001h
  call @ReadKey
  ; no key in buffer
  jz   @f
  ; ESC pressed - exit
  cmp  al, 01Bh
  jz   @f
  ; show pause text
  mov  dx, _wait
  call @ShowText
  ; wait for key
  mov  ax, 000h
  call @ReadKey
  push ax
  ; clear pause text
  mov  dx, _fill
  call @ShowText
  pop  ax
  ; ESC pressed - exit
  cmp  al, 01Bh
  jz   @f
  ; no error - continue printing
  xor  ax, ax
@@:
  ; if ESC pressed - close file
  cmp  al, 01Bh
  jnz  @f
  push ax
  ; close file handle since printing aborted (@SkipPage will be never called)
  call @FreeFile
  ; and delete partially saved file
  mov  dx, _name
  mov  ah, 041h
  int  021h
  pop  ax
@@:
  or   ax, ax
  retn

; -----------------------------------------------------------------------------

@ReadKey:
  ; ax = 0 - read, do not check
  test ax, ax
  jz   @f
  ; check for key in buffer
  mov  ah, 001h
  int  016h
  jnz  @f
  xor  ax, ax
  jmp  @ReadKey_ret
@@:
  ; read key from buffer
  mov  ah, 000h
  int  016h
@ReadKey_ret:
  or   ax, ax
  retn

; -----------------------------------------------------------------------------

@ShowText:
  push dx
  ; go to screen center
  mov  dx, 01814h
  xor  bh, bh
  mov  ah, 002h
  int  010h
  ; output string text
  pop  dx
  mov  ah, 009h
  int  021h
  retn

; -----------------------------------------------------------------------------

; data area
_name db 'PAGE0000.BMP',0
_wait db 'Pausing... Press a key to continue.$'
_fill db '                                   $'
; make everything below word-aligned
align 2
; DOS file handle
_file dw 0
; BITMAPFILEHEADER
_head            dw 04D42h ; _bfType
_bfSize_lo       dw 0      ; whole file size
_bfSize_hi       dw 0
_bfReserved1     dw 0
_bfReserved2     dw 0
_bfOffBits       dd HDRLEN ; offset to the bitmap data
; BITMAPINFOHEADER
_biSize          dd 40d    ; header size
_biWidth         dd 960d   ; page width
_biHeight_lo     dw 0      ; 760 / 1008 (NewsMaster 1.x / NewsMaster II)
_biHeight_hi     dw 0      ; if biHeight is negative, the bitmap is a top-down DIB
_biPlanes        dw 1
_biBitCount      dw 1
_biCompression   dd 0      ; 0 - BI_RGB - uncompressed
_biSizeImage_lo  dw 0      ; whole bitmap image data size
_biSizeImage_hi  dw 0
_biXPelsPerMeter dd 0
_biYPelsPerMeter dd 0
_biClrUsed       dd 0
_biClrImportant  dd 0
; palette
_palette dd 00000000h, 00FFFFFFh
HDRLEN = $ - _head
; buffer to hold one line of pixels (960 / 8 = 120)
_data db 120d dup(0)
